// 存储管理器 - 使用IndexedDB高性能本地存储
// 从Chrome Storage API迁移到IndexedDB
// 优势：无存储大小限制、事务支持、索引查询、高性能
// 注意：不支持跨设备同步，仅本地存储

// 导入IndexedDB存储管理器
// 注意：在实际使用中，需要确保storage-indexeddb.js已加载

// 如果需要从Chrome Storage迁移数据，请使用 DataMigrationManager
// 使用示例：
// const migrationManager = new DataMigrationManager();
// await migrationManager.quickMigration();

const StorageManager = {
  // 内部IndexedDB管理器实例
  _idbManager: null,
  
  // 初始化IndexedDB管理器
  async _getIDBManager() {
    if (!this._idbManager) {
      // 检查是否已有IndexedDBManager全局实例
      if (typeof IndexedDBManager !== 'undefined') {
        this._idbManager = new IndexedDBManager();
      } else {
        throw new Error('IndexedDBManager未加载，请确保storage-indexeddb.js已正确引入');
      }
    }
    
    await this._idbManager.init();
    return this._idbManager;
  },
  // 获取分类
  async getCategories() {
    const idbManager = await this._getIDBManager();
    return await idbManager.getCategories();
  },
  
  // 保存分类
  async saveCategories(categories) {
    // IndexedDB不支持批量替换，需要特殊处理
    console.warn('注意：saveCategories在IndexedDB中已被禁用，请使用addCategory/updateCategory方法');
    // 为了兼容性，这里不抛出错误，但不执行任何操作
    return Promise.resolve();
  },
  
  // 添加分类
  async addCategory(category) {
    const idbManager = await this._getIDBManager();
    return await idbManager.addCategory(category);
  },
  
  // 更新分类
  async updateCategory(categoryId, updates) {
    const idbManager = await this._getIDBManager();
    return await idbManager.updateCategory(categoryId, updates);
  },
  
  // 删除分类
  async deleteCategory(categoryId) {
    const idbManager = await this._getIDBManager();
    return await idbManager.deleteCategory(categoryId);
  },
  
  // 获取分类（包含链接数量）
  async getCategoriesWithCounts() {
    const idbManager = await this._getIDBManager();
    return await idbManager.getCategoriesWithCounts();
  },
  
  // 获取链接
  async getLinks() {
    const idbManager = await this._getIDBManager();
    return await idbManager.getLinks();
  },
  
  // 保存链接
  async saveLinks(links) {
    // IndexedDB不支持批量替换，需要特殊处理
    console.warn('注意：saveLinks在IndexedDB中已被禁用，请使用addLink/updateLink方法');
    // 为了兼容性，这里不抛出错误，但不执行任何操作
    return Promise.resolve();
  },
  
  // 添加链接
  async addLink(link) {
    const idbManager = await this._getIDBManager();
    return await idbManager.addLink(link);
  },
  
  // 更新链接
  async updateLink(linkId, updates) {
    const idbManager = await this._getIDBManager();
    return await idbManager.updateLink(linkId, updates);
  },
  
  // 记录链接访问
  async recordVisit(linkId) {
    const idbManager = await this._getIDBManager();
    return await idbManager.recordVisit(linkId);
  },
  
  // 生成Logo
  generateLogo(title) {
    const idbManager = this._idbManager;
    if (idbManager) {
      return idbManager.generateLogo(title);
    }
    
    // 备用实现
    if (!title) return '?';
    // 如果是中文，取第一个字
    if (/[一-龥]/.test(title)) {
      return title.charAt(0);
    }
    // 如果是英文，取前两个字母
    const words = title.split(' ');
    if (words.length >= 2) {
      return (words[0].charAt(0) + words[1].charAt(0)).toUpperCase();
    } else {
      return title.substring(0, 2).toUpperCase();
    }
  },
  
  // 删除链接
  async deleteLink(linkId) {
    const idbManager = await this._getIDBManager();
    return await idbManager.deleteLink(linkId);
  },
  
  // 获取设置
  async getSettings() {
    const idbManager = await this._getIDBManager();
    return await idbManager.getSettings();
  },
  
  // 保存设置
  async saveSettings(settings) {
    const idbManager = await this._getIDBManager();
    return await idbManager.saveSettings(settings);
  },
  
  // 更新设置
  async updateSettings(newSettings) {
    const idbManager = await this._getIDBManager();
    return await idbManager.updateSettings(newSettings);
  },
  
  // 获取所有数据
  async getAllData() {
    const idbManager = await this._getIDBManager();
    return await idbManager.getAllData();
  },
  
  // 导出数据
  async exportData() {
    const idbManager = await this._getIDBManager();
    return await idbManager.exportData();
  },
  
  // 导入数据
  async importData(data) {
    const idbManager = await this._getIDBManager();
    return await idbManager.importData(data);
  },
  
  // 更新分类顺序
  async updateCategoryOrder(categoryIds) {
    const idbManager = await this._getIDBManager();
    return await idbManager.updateCategoryOrder(categoryIds);
  },
  
  // 获取待办事项
  async getTodos() {
    const idbManager = await this._getIDBManager();
    return await idbManager.getTodos();
  },
  
  // 保存待办事项
  async saveTodos(todos) {
    const idbManager = await this._getIDBManager();
    return await idbManager.saveTodos(todos);
  },
  
  // 清空所有数据
  async clearAllData() {
    const idbManager = await this._getIDBManager();
    return await idbManager.clearAllData();
  }
};